#!/usr/local/bin/perl

#┌─────────────────────────────
#│ Diary Note : admin.cgi - 2021/02/13
#│ copyright (c) kentweb, 1997-2022
#│ https://www.kent-web.com/
#└─────────────────────────────

# モジュール宣言
use strict;
use CGI::Carp qw(fatalsToBrowser);
use vars qw(%in %cf);
use lib "./lib";
use CGI::Minimal;
use CGI::Session;
use Digest::SHA::PurePerl qw(sha256_base64);

# 設定ファイル認識
require "./init.cgi";
%cf = set_init();

# データ受理
CGI::Minimal::max_read_size($cf{maxdata});
my $cgi = CGI::Minimal->new;
error('容量オーバー') if ($cgi->truncated);
%in = parse_form($cgi);

# 認証
require "./lib/login.pl";
auth_login();

# 処理分岐
if ($in{year_log}) { year_log(); }
if ($in{data_mgr}) { data_mgr(); }
if ($in{pass_mgr}) { pass_mgr(); }
menu_list();

#-----------------------------------------------------------
#  管理画面
#-----------------------------------------------------------
sub menu_list {
	header("管理メニュー");
	print <<EOM;
<div id="body">
<p>選択ボタンを押してください。</p>
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<table class="form-tbl">
<tr>
	<th></th>
	<th>処理項目</th>
</tr><tr>
	<td><input type="submit" name="year_log" value="選択"></td>
	<td>年データの管理</td>
</tr><tr>
	<td><input type="submit" name="data_mgr" value="選択"></td>
	<td>個別記事の管理</td>
</tr><tr>
	<td><input type="submit" name="pass_mgr" value="選択"></td>
	<td>パスワード管理</td>
</tr><tr>
	<td><input type="submit" name="logoff" value="選択"></td>
	<td>ログアウト</td>
</tr>
</table>
</form>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  年データのメンテナンス
#-----------------------------------------------------------
sub year_log {
	# 新規西暦
	if ($in{year} && $in{add}) {
		if ($in{year} !~ /^\d{4}$/) {
			error("西暦は半角4桁で指定してください");
		}
		
		# 追加データ定義
		my (@data,@sort);
		$data[0] = "$in{year},0,0,0,0,0,0,0,0,0,0,0,0,\n";
		$sort[0] = $in{year};
		
		# index読み取り
		open(DAT,"+< $cf{datadir}/index.dat") or error("open err: index.dat");
		while (<DAT>) {
			my ($y) = split(/,/);
			
			push(@data,$_);
			push(@sort,$y);
		}
		
		# 降順ソート
		@data = @data[sort {$sort[$b] <=> $sort[$a]} 0 .. $#sort];
		
		# index更新
		seek(DAT, 0, 0);
		print DAT @data;
		truncate(DAT, tell(DAT));
		close(DAT);
	
	# 西暦削除（確認）
	} elsif ($in{pre} && $in{del}) {
		
		header("削除確認");
		print qq|<div id="body">\n|;
		print qq|▼<b>$in{pre}年のデータを削除します。</b>\n|;
		print qq|<p>$in{pre}年の全ての本文と画像を削除します。<br>\n|;
		print qq|いったん削除すると復元はできません。</p>\n|;
		print qq|<form action="$cf{admin_cgi}" method="post">\n|;
		print qq|<input type="hidden" name="year_log" value="1">\n|;
		print qq|<input type="hidden" name="sid" value="$in{sid}">\n|;
		print qq|<input type="hidden" name="del" value="$in{pre}">\n|;
		print qq|<input type="submit" value="一括削除する"></form>\n|;
		print qq|</div>\n|;
		print qq|</body></html>\n|;
		exit;
	
	# 西暦削除（実行）
	} elsif ($in{del}) {
		
		# index更新
		my @data;
		open(DAT,"+< $cf{datadir}/index.dat") or error("open err: index.dat");
		while (<DAT>) {
			my ($y) = split(/,/);
			next if ($in{del} == $y);
			
			push(@data,$_);
		}
		seek(DAT, 0, 0);
		print DAT @data;
		truncate(DAT, tell(DAT));
		close(DAT);
		
		# 日記データ削除
		foreach my $i (1 .. 12) {
			$i = sprintf("%02d",$i);
			next unless (-e "$cf{datadir}/log/$in{del}$i.txt");
			
			open(IN,"$cf{datadir}/log/$in{del}$i.txt");
			while (<IN>) {
				my ($no,$d,$w,$sub,$msg,$e1,$w1,$h1,$e2,$w2,$h2,$e3,$w3,$h3) = split(/<>/);
				
				# 添付画像削除
				if ($e1) {
					unlink("$cf{imgdir}/$no-1$e1");
					unlink("$cf{imgdir}/$no-s-1$e1") if (-f "$cf{imgdir}/$no-s-1$e1");
				}
				if ($e2) {
					unlink("$cf{imgdir}/$no-2$e2");
					unlink("$cf{imgdir}/$no-s-2$e2") if (-f "$cf{imgdir}/$no-s-2$e2");
				}
				if ($e3) {
					unlink("$cf{imgdir}/$no-3$e3");
					unlink("$cf{imgdir}/$no-s-3$e3") if (-f "$cf{imgdir}/$no-s-3$e3");
				}
			}
			close(IN);
			
			# 月次データ削除
			unlink("$cf{datadir}/log/$in{del}$i.txt");
		}
	}
	
	# メニュー画面
	header("年データの管理");
	back_btn();
	print <<EOM;
<p class="ttl">■ 年データの管理</p>
<div id="body">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="year_log" value="1">
<input type="hidden" name="sid" value="$in{sid}">
<p><b>・ 新規に年データを作成</b></p>
<table class="form-tbl">
<tr>
	<th>西暦（数字4桁）</th>
</tr><tr>
	<td class="ta-c">
		<input type="text" name="year" size="6" style="ime-mode:inactive" maxlength="4">
		年 &nbsp;
		<input type="submit" name="add" value="新規生成">
	</td>
</tr>
</table>
<p><b>・ 年データを一括削除</b></p>
<table class="form-tbl">
<tr>
	<th>選択</th>
	<th>年データ</th>
</tr>
EOM

	open(IN,"$cf{datadir}/index.dat") or error("open err: index.dat");
	while(<IN>) {
		my ($y) = (split(/,/))[0];
		
		print qq|<tr><td class="ta-c"><input type="radio" name="pre" value="$y"></td>|;
		print qq|<td>$y年</td></tr>\n|;
	}
	close(IN);
	
	print <<EOM;
</table>
<input type="submit" name="del" value="一括削除">
</form>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  個別記事の書込＆メンテナンス
#-----------------------------------------------------------
sub data_mgr {
	if ($in{ope} eq 'diary_log') { diary_log(); }
	
	# 現在
	my ($mon,$year) = (localtime())[4,5];
	$year += 1900;
	$mon++;
	
	# メニュー画面
	header("個別記事の管理");
	back_btn();
	print <<EOM;
<p class="ttl">■ 個別記事の管理</p>
<div id="body">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="data_mgr" value="1">
<input type="hidden" name="ope" value="diary_log">
<input type="hidden" name="sid" value="$in{sid}">
<p>
	・ 書込又は修正を行う年月を選択してください。<br>
	・ 新しい年の記事を書く場合は「年データの管理」から最初にデータを作成してください。<br>
	・ 枠内の数字は月単位の記事数です。
</p>
<table class="form-tbl">
<tr>
	<th>&nbsp;</th>
EOM

	for (1 .. 12) {
		print qq|<td align="center">$_月</td>|;
	}
	print "</tr>\n";
	
	# index読み込み
	open(IN,"$cf{datadir}/index.dat") or error("open err: index.dat");
	while(<IN>) {
		my ($y,@mon) = split(/,/);
		
		print qq|<tr><th>$y年</th>|;
		
		for my $i (1 .. 12) {
			$i = sprintf("%02d",$i);
			
			my $n = 0;
			open(DB,"$cf{datadir}/log/$y$i.txt");
			while( my $log = <DB> ) {
				my ($no,$d,$w,$sub) = (split(/<>/,$log))[0..3];
				
				if ($sub) { $n++; }
			}
			close(DB);
			
			print $year == $y && $mon == $i ? qq|<td width="40" style="background:#ffd7de">| : qq|<td width="40">|;
			print qq|<input type="radio" name="ym" value="$y$i"> $n</td>|;
		}
		print "</tr>\n";
	}
	close(IN);
	
	print <<EOM;
</table>
<input type="submit" name="submit" value="送信する">
</form>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  個別記事
#-----------------------------------------------------------
sub diary_log {
	my ($y,$m);
	if ($in{y} =~ /^\d{4}$/ && $in{m} =~ /^\d{2}$/) {
		$y = $in{y};
		$m = $in{m};
	} elsif ($in{ym} =~ /^(\d{4})(\d{2})$/) {
		$y = $1;
		$m = $2;
	} else {
		error("選択年月が不正です");
	}
	
	# 削除
	if ($in{job} eq "dele" && $in{no}) {
		
		# 削除情報
		my %del;
		foreach ( split(/\0/,$in{no}) ) { $del{$_}++; }
		
		my @data;
		open(DAT,"+< $cf{datadir}/log/$in{y}$in{m}.txt") or error("open err: $in{y}$in{m}.txt");
		while (<DAT>) {
			my ($no,$d,$w,$sub,$msg,$e1,$w1,$h1,$e2,$w2,$h2,$e3,$w3,$h3) = split(/<>/);
			
			if (defined($del{$no})) {
				unlink("$cf{imgdir}/$no-1$e1") if (-f "$cf{imgdir}/$no-1$e1");
				unlink("$cf{imgdir}/$no-2$e2") if (-f "$cf{imgdir}/$no-2$e2");
				unlink("$cf{imgdir}/$no-3$e3") if (-f "$cf{imgdir}/$no-3$e3");
				unlink("$cf{imgdir}/$no-s-1$e1") if (-f "$cf{imgdir}/$no-s-1$e1");
				unlink("$cf{imgdir}/$no-s-2$e2") if (-f "$cf{imgdir}/$no-s-2$e2");
				unlink("$cf{imgdir}/$no-s-3$e3") if (-f "$cf{imgdir}/$no-s-3$e3");
				next;
			}
			push(@data,$_);
		}
		
		# 更新
		if (@data == 0) {
			close(DAT);
			unlink("$cf{datadir}/log/$in{y}$in{m}.txt");
			
			# index更新
			my @file;
			open(DAT,"+< $cf{datadir}/index.dat") or error("open err: index.dat");
			while (<DAT>) {
				my @f = split(/,/);
				
				if ($in{y} == $f[0]) {
					my $new;
					foreach my $f (0 .. 12) {
						if ($f == $in{m}) { $f[$f] = 0; }
						$new .= "$f[$f],";
					}
					$_ = "$new\n";
				}
				push(@file,$_);
			}
			seek(DAT, 0, 0);
			print DAT @file;
			truncate(DAT, tell(DAT));
			close(DAT);
		
		} else {
			seek(DAT, 0, 0);
			print DAT @data;
			truncate(DAT, tell(DAT));
			close(DAT);
		}
		
		$y = $in{y};
		$m = $in{m};
	
	# 新規画面
	} elsif ($in{job} eq "new") {
		
		new_form();
	
	# 新規書き込み
	} elsif ($in{job} eq "new2") {
		
		# 入力チェック
		if ($in{sub} eq "" || $in{comment} eq "") {
			error("タイトル又は内容が未入力です");
		}
		
		# 記事NO取得
		open(DAT,"+< $cf{datadir}/num.dat") or error("open err: num.dat");
		my $num = <DAT> + 1;
		seek(DAT, 0, 0);
		print DAT $num;
		truncate(DAT, tell(DAT));
		close(DAT);
		
		$in{d} = sprintf("%02d",$in{d});
		
		# 月で新規のときログを生成する
		if (! -e "$cf{datadir}/log/$in{y}$in{m}.txt") {
			open(DAT,"+> $cf{datadir}/log/$in{y}$in{m}.txt");
			close(DAT);
			chmod(0666,"$cf{datadir}/log/$in{y}$in{m}.txt");
		}
		
		my ($i,@data,@tmp1,@tmp2);
		open(DAT,"+< $cf{datadir}/log/$in{y}$in{m}.txt");
		while (<DAT>) {
			my ($no,$d) = split(/<>/);
			
			$i++;
			push(@tmp1,$d);
			push(@tmp2,$i);
			push(@data,$_);
		}
		
		# 画像アップ
		my ($t1,$w1,$h1,$t2,$w2,$h2,$t3,$w3,$h3);
		if ($in{upfile1} || $in{upfile2} || $in{upfile3}) {
			($t1,$w1,$h1,$t2,$w2,$h2,$t3,$w3,$h3) = upload($num);
		}
		
		# 曜日算出
		my $w = get_week($in{y},$in{m},$in{d});
		
		# 記事追加
		$i++;
		push(@data,"$num<>$in{d}<>$w<>$in{sub}<>$in{comment}<>$t1<>$w1<>$h1<>$t2<>$w2<>$h2<>$t3<>$w3<>$h3<>\n");
		push(@tmp1,$in{d});
		push(@tmp2,$i);
		
		# 記事ソート
		if ($cf{sort_day}) {
			# 降順
			@data = @data[sort{$tmp1[$b] <=> $tmp1[$a] || $tmp2[$b] <=> $tmp2[$a]} 0 .. $#tmp1];
		} else {
			# 昇順
			@data = @data[sort{$tmp1[$a] <=> $tmp1[$b] || $tmp2[$b] <=> $tmp2[$a]} 0 .. $#tmp1];
		}
		
		# データ更新
		seek(DAT, 0, 0);
		print DAT @data;
		truncate(DAT, tell(DAT));
		close(DAT);
		
		# index更新
		my $flg;
		@data = ();
		open(DAT,"+< $cf{datadir}/index.dat") or error("open err: index.dat");
		while (<DAT>) {
			my @f = split(/,/);
			
			if ($in{y} == $f[0]) {
				if ($f[$in{m}] == 1) { $flg++; last; }
				my $new;
				foreach my $f (0 .. 12) {
					if ($f == $in{m}) { $f[$f] = 1; }
					$new .= "$f[$f],";
				}
				$_ = "$new\n";
			}
			push(@data,$_);
		}
		
		if (!$flg) {
			seek(DAT, 0, 0);
			print DAT @data;
			truncate(DAT, tell(DAT));
		}
		close(DAT);
		
		$y = $in{y};
		$m = $in{m};
	
	# 修正画面
	} elsif ($in{job} eq "edit" && $in{no}) {
		
		if ($in{no} =~ /\0/) { error("修正の選択は１つのみです"); }
		
		my @f;
		open(IN,"$cf{datadir}/log/$in{y}$in{m}.txt") or error("open err: $in{y}$in{m}.txt");
		while (<IN>) {
			@f = split(/<>/);
			
			last if ($in{no} == $f[0]);
		}
		close(IN);
		
		edit_form(@f);
	
	# 修正実行
	} elsif ($in{job} eq "edit2") {
		
		# 入力チェック
		if ($in{sub} eq "" || $in{comment} eq "") {
			error("タイトル又は内容が未入力です");
		}
		
		# 画像アップ
		my (@t2,@w2,@h2);
		if ($in{upfile1} || $in{upfile2} || $in{upfile3}) {
			($t2[0],$w2[0],$h2[0],$t2[1],$w2[1],$h2[1],$t2[2],$w2[2],$h2[2]) = upload($in{no});
		}
		
		my @data;
		open(DAT,"+< $cf{datadir}/log/$in{y}$in{m}.txt") or error("open err: $in{y}$in{m}.txt");
		while (<DAT>) {
			my ($no,$d,$w,$sub,$msg,$e1,$w1,$h1,$e2,$w2,$h2,$e3,$w3,$h3) = split(/<>/);
			my @t = ($e1,$e2,$e3);
			my @w = ($w1,$w2,$w3);
			my @h = ($h1,$h2,$h3);
			
			if ($in{no} == $no) {
				
				# 画像
				foreach my $i (0 .. 2) {
					my $j = $i + 1;
					
					# 新規追加のとき
					if ($t2[$i] && !$t[$i]) {
						$t[$i] = $t2[$i];
						$w[$i] = $w2[$i];
						$h[$i] = $h2[$i];
					
					# 差替のとき
					} elsif ($t2[$i] && $t[$i]) {
						
						# 拡張子変更のとき
						if ($t[$i] ne $t2[$i]) {
							unlink("$cf{imgdir}/$in{no}-$j$t[$i]");
							unlink("$cf{imgdir}/$in{no}-s-$j$t[$i]") if (-f "$cf{imgdir}/$in{no}-s-$j$t[$i]");
						}
						
						$t[$i] = $t2[$i];
						$w[$i] = $w2[$i];
						$h[$i] = $h2[$i];
					
					# 削除のとき
					} elsif ($in{"imgdel$j"} == 1) {
						unlink("$cf{imgdir}/$in{no}-$j$t[$i]");
						unlink("$cf{imgdir}/$in{no}-s-$j$t[$i]") if (-f "$cf{imgdir}/$in{no}-s-$j$t[$i]");
						$t[$i] = "";
						$w[$i] = "";
						$h[$i] = "";
					}
				}
				$_ = "$no<>$d<>$w<>$in{sub}<>$in{comment}<>$t[0]<>$w[0]<>$h[0]<>$t[1]<>$w[1]<>$h[1]<>$t[2]<>$w[2]<>$h[2]<>\n";
			}
			push(@data,$_);
		}
		
		# ログ更新
		seek(DAT, 0, 0);
		print DAT @data;
		truncate(DAT, tell(DAT));
		close(DAT);
		
		$y = $in{y};
		$m = $in{m};
	}
	
	if ($y eq "") { $y = $in{y}; }
	if ($m eq "") { $m = $in{m}; }
	
	# 初期画面
	header("個別記事の管理 &gt; 編集");
	back_btn();
	print <<EOM;
<p class="ttl">■ 個別記事の管理 &gt; 編集</p>
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="data_mgr" value="1">
<input type="hidden" name="ope" value="diary_log">
<input type="hidden" name="y" value="$y">
<input type="hidden" name="m" value="$m">
<input type="hidden" name="sid" value="$in{sid}">
<div id="body">
<p>
・ 処理を選択して送信ボタンを押してください。<br>
・ 修正又は削除の場合は、記事にチェックを入れてください。
</p>
処理：
<select name="job">
<option value="new">新規
<option value="edit">修正
<option value="dele">削除
</select>
<input type="submit" value="送信する">
<table class="form-tbl">
<tr>
	<th>選択</th>
	<th>年月日</th>
	<th>タイトル</th>
</tr>
EOM

	# 記事展開
	open(IN,"$cf{datadir}/log/$y$m.txt");
	while (<IN>) {
		my ($no,$d,$w,$sub,$msg,$e1,$w1,$h1,$e2,$w2,$h2,$e3,$w3,$h3) = split(/<>/);
		
		# 表示
		print qq|<tr><td class="ta-c"><input type="checkbox" name="no" value="$no"></td>\n|;
		print qq|<td>$y/$m/$d ($cf{week}[$w])</td>\n|;
		print qq|<td><b>$sub</b></td></tr>\n|;
	}
	close(IN);
	
	print <<EOM;
</table>
</form>
</div>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  新規画面
#-----------------------------------------------------------
sub new_form {
	header("日記書き込みフォーム");
	print <<EOM;
<div align="right">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="data_mgr" value="1">
<input type="hidden" name="ope" value="diary_log">
<input type="hidden" name="y" value="$in{y}">
<input type="hidden" name="m" value="$in{m}">
<input type="submit" value="前に戻る">
</form>
</div>
<p class="ttl">■新規登録フォーム</p>
<div id="body">
<p>・ 書込みを行う「日」を選択してください。</p>
<form action="$cf{admin_cgi}" method="post" enctype="multipart/form-data">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="data_mgr" value="1">
<input type="hidden" name="ope" value="diary_log">
<input type="hidden" name="job" value="new2">
<input type="hidden" name="y" value="$in{y}">
<input type="hidden" name="m" value="$in{m}">
<table class="form-tbl">
<tr>
	<th>年月日</th>
	<td>$in{y}年$in{m}月
		<select name="d">
EOM

	# 当該月の末日算出
	my $last = (31,28,31,30,31,30,31,31,30,31,30,31) [$in{m}-1]
	+ ($in{m} == 2 && (($in{y} % 4 == 0 && $in{y} % 100 != 0) ||
	$in{y} % 400 == 0));

	foreach (1 .. $last) {
		print "<option value=\"$_\">$_日\n";
	}

	print <<EOM;
  </select></td>
</tr><tr>
  <th>タイトル</th>
  <td><input type="text" name="sub" size="30"></td>
</tr><tr>
  <th>内容</th>
  <td><textarea name="comment" cols="45" rows="6"></textarea></td>
</tr><tr>
  <th>画像1</th>
  <td><input type="file" name="upfile1" size="50"></td>
</tr><tr>
  <th>画像2</th>
  <td><input type="file" name="upfile2" size="50"></td>
</tr><tr>
  <th>画像3</th>
  <td><input type="file" name="upfile3" size="50"></td>
</tr>
</table>
<input type="submit" value="送信する">
</form>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  修正画面
#-----------------------------------------------------------
sub edit_form {
	my ($no,$d,$w,$sub,$msg,$e1,$w1,$h1,$e2,$w2,$h2,$e3,$w3,$h3) = @_;
	my @t = ($e1,$e2,$e3);
	my @w = ($w1,$w2,$w3);
	my @h = ($h1,$h2,$h3);
	$msg =~ s|<br( /)?>|\n|g;

	header("修正フォーム");
	print <<EOM;
<div align="right">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="data_mgr" value="1">
<input type="hidden" name="ope" value="diary_log">
<input type="hidden" name="y" value="$in{y}">
<input type="hidden" name="m" value="$in{m}">
<input type="submit" value="前に戻る">
</form>
</div>
<p class="ttl">■修正フォーム</p>
<div id="body">
<form action="$cf{admin_cgi}" method="post" enctype="multipart/form-data">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="data_mgr" value="1">
<input type="hidden" name="ope" value="diary_log">
<input type="hidden" name="job" value="edit2">
<input type="hidden" name="y" value="$in{y}">
<input type="hidden" name="m" value="$in{m}">
<input type="hidden" name="d" value="$in{d}">
<input type="hidden" name="no" value="$in{no}">
<p>
・ 変更する部分のみを修正してください。<br>
・ 画像を差替える場合又は追加する場合は、各フィールドに画像を指定してください。<br>
・ 画像を削除する場合は、チェックボックスにチェックを入れてください。<br>
・ 画像はGIF/JPEG/PNG形式のみ有効です。
</p>
<table class="form-tbl">
<tr>
	<th>年月日</th>
	<td><b>$in{y}/$in{m}/$d ($cf{week}->[$w])</b></td>
</tr><tr>
	<th>タイトル</th>
	<td><input type="text" name="sub" size="30" value="$sub"></td>
</tr><tr>
	<th>内容</th>
	<td><textarea name="comment" cols="45" rows="6">$msg</textarea></td>
</tr>
EOM

	foreach my $i (0 .. 2) {
		my $j = $i + 1;
		
		print qq|<tr><th>画像$j</th>|;
		print qq|<td><input type="file" name="upfile$j" size="50">\n|;
		
		if ($t[$i]) {
			print qq|&nbsp;[<a href="$cf{imgurl}/$in{no}-$j$t[$i]">画像</a>]\n|;
			print qq|<input type="checkbox" name="imgdel$j" value="1">削除|;
		}
		
		print "</td></tr>\n";
	}
	
	print <<EOM;
</table>
<input type="submit" value="送信する">
</form>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  画像アップロード
#-----------------------------------------------------------
sub upload {
	my $no = shift;
	
	# サムネイル機能
	require './lib/thumb.pl' if ($cf{thumbnail});
	
	my @ret;
	foreach my $i (1 .. 3) {
		
		# ファイル名
		my $fname = $cgi->param_filename("upfile$i");
		if ($fname =~ /(\.jpe?g|\.png|\.gif)$/i) {
			my $ex = $1;
			if ($ex eq '.jpeg') { $ex = '.jpg'; }
			
			# アップファイル定義
			my $imgfile = "$cf{imgdir}/$no-$i$ex";
			
			# 書き込み
			open(OUT,"+> $imgfile") or error("画像アップ失敗");
			binmode(OUT);
			print OUT $in{"upfile$i"};
			close(OUT);
			
			# 画像サイズ取得
			my ($w,$h);
			if ($ex eq ".jpg") { ($w,$h) = j_size($imgfile); }
			elsif ($ex eq ".gif") { ($w,$h) = g_size($imgfile); }
			elsif ($ex eq ".png") { ($w,$h) = p_size($imgfile); }
			
			# サムネイル作成
			if ($cf{thumbnail}) {
				($w,$h) = resize($w,$h);
				my $thumb = "$cf{imgdir}/$no-s-$i$ex";
				make_thumb($imgfile,$thumb,$w,$h);
			}
			
			push(@ret,($ex,$w,$h));
		} else {
			push(@ret,('','',''));
		}
	}
	# 返り値
	return @ret;
}

#-----------------------------------------------------------
#  JPEGサイズ認識
#-----------------------------------------------------------
sub j_size {
	my $jpg = shift;
	
	my ($h, $w, $t);
	open(IMG,"$jpg");
	binmode(IMG);
	read(IMG, $t, 2);
	while (1) {
		read(IMG, $t, 4);
		my ($m, $c, $l) = unpack("a a n", $t);
		
		if ($m ne "\xFF") {
			$w = $h = 0;
			last;
		} elsif ((ord($c) >= 0xC0) && (ord($c) <= 0xC3)) {
			read(IMG, $t, 5);
			($h, $w) = unpack("xnn", $t);
			last;
		} else {
			read(IMG, $t, ($l - 2));
		}
	}
	close(IMG);
	
	return ($w, $h);
}

#-----------------------------------------------------------
#  GIFサイズ認識
#-----------------------------------------------------------
sub g_size {
	my $gif = shift;
	
	my $data;
	open(IMG,"$gif");
	binmode(IMG);
	sysread(IMG, $data, 10);
	close(IMG);
	
	if ($data =~ /^GIF/) { $data = substr($data, -4); }
	my $w = unpack("v", substr($data, 0, 2));
	my $h = unpack("v", substr($data, 2, 2));
	
	return ($w, $h);
}

#-----------------------------------------------------------
#  PNGサイズ認識
#-----------------------------------------------------------
sub p_size {
	my $png = shift;
	
	my $data;
	open(IMG,"$png");
	binmode(IMG);
	read(IMG, $data, 24);
	close(IMG);
	
	my $w = unpack("N", substr($data, 16, 20));
	my $h = unpack("N", substr($data, 20, 24));
	
	return ($w, $h);
}

#-----------------------------------------------------------
#  HTMLヘッダー
#-----------------------------------------------------------
sub header {
	my $ttl = shift;
	
	print <<EOM;
Content-type: text/html; charset=utf-8

<!doctype html>
<html lang="ja">
<head>
<meta charset="UTF-8">
<link href="$cf{cmnurl}/admin.css" rel="stylesheet">
<title>$ttl</title>
</head>
<body>
<div id="head">
	<img src="$cf{cmnurl}/star.png" alt="star" class="icon">
	Diary Note 管理画面 ::
</div>
EOM
}

#-----------------------------------------------------------
#  戻りボタン
#-----------------------------------------------------------
sub back_btn {
	print <<EOM;
<div class="back-btn">
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="submit" value="▲メニュー">
</form>
</div>
EOM
}

#-----------------------------------------------------------
#  完了画面
#-----------------------------------------------------------
sub message {
	my $msg = shift;
	
	header($msg);
	print <<EOM;
<h3>処理完了</h3>
<font color="#dd0000" size="-1">$msg</font>
<form action="$cf{admin_cgi}" method="post">
<input type="hidden" name="sid" value="$in{sid}">
<input type="hidden" name="mode" value="$in{mode}">
<input type="submit" value="元の画面に戻る">
</form>
</body>
</html>
EOM
	exit;
}

#-----------------------------------------------------------
#  エラー処理
#-----------------------------------------------------------
sub error {
	my $err = shift;
	
	header("ERROR");
	print <<EOM;
<h3>ERROR !</h3>
<p class="err">$err</p>
<p><input type="button" value="前画面に戻る" onclick="history.back()"></p>
</body>
</html>
EOM
	exit;
}

