#!/usr/local/bin/perl

#┌─────────────────────────────
#│ Diary Note : diary.cgi - 2021/03/28
#│ copyright (c) kentweb, 1997-2022
#│ https://www.kent-web.com/
#└─────────────────────────────

# モジュール宣言
use strict;
use CGI::Carp qw(fatalsToBrowser);
use lib "./lib";
use CGI::Minimal;
use Calendar::Japanese::Holiday;

# 設定ファイル認識
require "./init.cgi";
my %cf = set_init();

# データ受理
CGI::Minimal::max_read_size($cf{maxdata});
my $cgi = CGI::Minimal->new;
error('容量オーバー') if ($cgi->truncated);
my %in = parse_form($cgi);

# 処理分岐
if ($in{mode} eq "read") { read_log(); }
if ($in{mode} eq "pop")  { pop_day(); }
top_menu();

#-----------------------------------------------------------
#  TOP画面
#-----------------------------------------------------------
sub top_menu {
	# メッセージ
	$cf{message} =~ s/\n/<br>/g;
	
	# テンプレート読み込み
	open(IN,"$cf{tmpldir}/top.html") or error("open err: top.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	# 置き換え
	$tmpl =~ s/!(cgi_title|homepage|cmnurl)!/$cf{$1}/g;
	$tmpl =~ s/<!-- message -->/$cf{message}/;
	
	# テンプレート分割
	my ($head,$loop,$foot) = $tmpl =~ m|(.+)<!-- loop -->(.+?)<!-- /loop -->(.+)|s
			? ($1,$2,$3)
			: error("テンプレート不正");
	
	# ヘッダ
	print "Content-type: text/html; charset=utf-8\n\n";
	print $head;
	
	# index読み取り
	open(IN,"$cf{datadir}/index.dat") or error("open err: index.dat");
	while (<IN>) {
		my @f = split(/,/);
		
		my $tmp = $loop;
		$tmp =~ s/!year!/$f[0]/g;
		
		for my $i (1 .. 12) {
			if ($f[$i]) {
				$tmp =~ s|!$i!|<a href="$cf{diary_cgi}?mode=read&amp;y=$f[0]&amp;m=$i">$i月</a>|g;
			} else {
				$tmp =~ s|!$i!|$i月|g;
			}
		}
		
		print $tmp;
	}
	close(IN);
	
	# フッタ
	footer($foot);
}

#-----------------------------------------------------------
#  内容閲覧
#-----------------------------------------------------------
sub read_log {
	# 現在の年月
	my ($mday,$mon,$year) = (localtime())[3 .. 5];
	$year += 1900;
	$mon++;
	$in{y} =~ s/\D//g;
	$in{m} =~ s/\D//g;
	if ($in{y} < 1970 or $in{m} < 1 or $in{m} > 12) {
		$in{y} = $year;
		$in{y} = $mon;
	}
	
	# ページ数
	my $pg = $in{pg} || 0;
	
	# テンプレート読込
	open(IN,"$cf{tmpldir}/read.html") or error("open err: read.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	# 文字置き換え
	$tmpl =~ s/!(cgi_title|cmnurl)!/$cf{$1}/g;
	$tmpl =~ s/!year!/$in{y}/g;
	$tmpl =~ s/!month!/$in{m}/g;
	$tmpl =~ s/!([a-z]+_cgi)!/$cf{$1}/g;
	$tmpl =~ s|!icon:(\w+\.\w+)!|<img src="$cf{cmnurl}/$1" alt="$1" class="icon">|g;
	
	
	# テンプレート分割
	my ($head,$loop,$foot) = $tmpl =~ m|(.+)<!-- loop -->(.+?)<!-- /loop -->(.+)|s
				? ($1,$2,$3)
				: error("テンプレート不正");
	
	# ログ展開（ログが存在しない場合はエラーにしない）
	my $log = sprintf("%04d%02d",$in{y},$in{m});
	my ($body,$i,%cal);
	open(IN,"$cf{datadir}/log/$log.txt");
	while (<IN>) {
		$i++;
		my ($no,$d,$wk,$sub,$msg,$e1,$w1,$h1,$e2,$w2,$h2,$e3,$w3,$h3) = split(/<>/);
		$cal{$d}++;
		next if ($i < $pg + 1);
		next if ($i > $pg + $cf{pg_max});
		
		# 自動リンク
		$msg = auto_link($msg) if ($cf{autolink});
		
		# 画像
		my $image;
		if ($e1) { $image .= image(1,$no,$e1,$w1,$h1); }
		if ($e2) { $image .= image(2,$no,$e2,$w2,$h2); }
		if ($e3) { $image .= image(3,$no,$e3,$w3,$h3); }
		
		my $day = $d;
		$day =~ s/^0//;
		
		my $tmp = $loop;
		$tmp =~ s/!date!/$in{y}.$in{m}.$day/g;
		$tmp =~ s/!day-title!/$sub/g;
		$tmp =~ s/!comment!/$msg/g;
		$tmp =~ s/!image!/$image/g;
		
		$body .= $tmp;
	}
	close(IN);
	
	$foot =~ s/!page-btn!/make_pgbtn($i,$pg)/e;
	$foot =~ s|<!-- calen -->(.+?)<!-- /calen -->|put_calen($1,$mday,$mon,$year,\%cal)|se;
	
	# 先月/翌月ボタン
	my ($b_y,$b_m,$n_y,$n_m) = get_ym($in{y},$in{m});
	$foot =~ s|!back-btn!|<a href="$cf{diary_cgi}?mode=read&amp;y=$b_y&amp;m=$b_m"><img src="$cf{cmnurl}/back.gif" alt="前月"></a>|;
	$foot =~ s|!next-btn!|<a href="$cf{diary_cgi}?mode=read&amp;y=$n_y&amp;m=$n_m"><img src="$cf{cmnurl}/next.gif" alt="翌月"></a>|;
	
	# 表示
	print "Content-type: text/html; charset=utf-8\n\n";
	print $head, $body;
	footer($foot);
}

#-----------------------------------------------------------
#  カレンダ
#-----------------------------------------------------------
sub put_calen {
	my ($loop,$mday,$mon,$year,$cal) = @_;
	
	# 1日の曜日を求める
	my $week = get_week($in{y},$in{m},1);
	
	# 末日を求める
	my $last = last_day($in{y},$in{m});
	
	# 祝日を認識
	my $hdays = eval "getHolidays($in{y},$in{m},1);";
	
	my ($ret,$day,$end);
	for my $k (1 .. 6) {
		
		# 週のテンプレート情報
		my $tmp = $loop;
		
		# 末日のフラグが立った場合はループを終了
		last if ($end);
		
		# 日から土まで7日分をループ
		for my $i (0 .. 6) {
			
			# 第１週で初日の週より小のとき、又は終了フラグが立ったときは「空き枠」
			if (($k == 1 && $i < $week) || $end) {
				$tmp =~ s/!$i!/&nbsp;/;
			
			# 実枠あり
			} else {
				# 日を数える
				$day++;
				
				# 枠処理
				my $d = sprintf("%02d",$day);
				my $col  = defined $$hdays{$d} ? $cf{week_col}[7] : $cf{week_col}[$i];
				my $cday = defined $$cal{$d}
						? qq|<a href="$cf{diary_cgi}?mode=pop&amp;y=$in{y}&amp;m=$in{m}&amp;d=$day" target="popwin" onclick="popup(0)" style="color:$col">$day</a>|
						: qq|<span style="color:$col">$day</span>|;
				
				# 本日は背景色を変える
				if ($in{y} == $year && $in{m} == $mon && $day == $mday) {
					$cday = qq|<div style="background:$cf{cal_today}">$cday</div>|;
				}
				
				$tmp =~ s/!$i!/$cday/;
			}
			
			# 末日に達した場合フラグを立てる
			if ($day >= $last) { $end = 1; }
		}
		$ret .= $tmp;
	}
	return $ret;
}

#-----------------------------------------------------------
#  前後の年月を求める
#-----------------------------------------------------------
sub get_ym {
	my ($y,$m) = @_;
	
	# 前月
	my ($b_y,$b_m);
	if ($m == 1) {
		$b_y = $y - 1;
		$b_m = 12;
	} else {
		$b_y = $y;
		$b_m = $m - 1;
	}
	# 翌月
	my ($n_y,$n_m);
	if ($m == 12) {
		$n_y = $y + 1;
		$n_m = 1;
	} else {
		$n_y = $y;
		$n_m = $m + 1;
	}
	return ($b_y,$b_m,$n_y,$n_m);
}

#-----------------------------------------------------------
#  ポップアップ画面
#-----------------------------------------------------------
sub pop_day {
	# 汚染チェック
	$in{y} =~ s/\D//g;
	$in{m} =~ s/\D//g;
	$in{d} =~ s/\D//g;
	
	# データファイル
	my $log = sprintf("%04d%02d",$in{y},$in{m});
	
	# 該当記事抽出
	my @log;
	open(IN,"$cf{datadir}/log/$log.txt") or error("open err: $log.txt");
	while (<IN>) {
		my ($d) = (split(/<>/))[1];
		
		if ($in{d} == $d) {
			push(@log,$_);
		}
	}
	close(IN);
	
	if (@log == 0) { error("該当記事なし"); }
	
	# テンプレート読込
	open(IN,"$cf{tmpldir}/popup.html") or error("open err: popup.html");
	my $tmpl = join('',<IN>);
	close(IN);
	
	# 置き換え
	$tmpl =~ s/!(cgi_title|cmnurl)!/$cf{$1}/g;
	
	# テンプレート分割
	my ($head,$loop,$foot) = $tmpl =~ m|(.+)<!-- loop -->(.+?)<!-- /loop -->(.+)|s
			? ($1,$2,$3)
			: error("テンプレート不正");
	
	# ヘッダ
	print "Content-type: text/html; charset=utf-8\n\n";
	print $head;
	
	# 表示開始
	for (@log) {
		my ($no,$d,$wk,$sub,$msg,$e1,$w1,$h1,$e2,$w2,$h2,$e3,$w3,$h3) = split(/<>/);
		
		# 自動リンク
		$msg = auto_link($msg) if ($cf{autolink});
		
		# 画像
		my $image;
		if ($e1) { $image .= image(1,$no,$e1,$w1,$h1); }
		if ($e2) { $image .= image(2,$no,$e2,$w2,$h2); }
		if ($e3) { $image .= image(3,$no,$e3,$w3,$h3); }
		
		my $day = $d;
		$day =~ s/^0//;
		
		my $tmp = $loop;
		$tmp =~ s/!date!/$in{m}月$day日（$cf{week}[$wk]）/g;
		$tmp =~ s/!day-title!/$sub/g;
		$tmp =~ s/!comment!/$msg/g;
		$tmp =~ s/!image!/$image/g;
		print $tmp;
	}
	
	print $foot;
	exit;
}

#-----------------------------------------------------------
#  フッター
#-----------------------------------------------------------
sub footer {
	my $foot = shift;
	
	# 著作権表記（削除・改変禁止）
	my $copy = <<EOM;
<p style="margin-top:2em;text-align:center;font-size:10px;" class="copy">
	- <a href="https://www.kent-web.com/" target="_top">DiaryNote</a> -
</p>
EOM

	if ($foot =~ /(.+)(<\/body[^>]*>.*)/si) {
		print "$1$copy$2\n";
	} else {
		print "$foot$copy\n";
		print "</body></html>\n";
	}
	exit;
}

#-----------------------------------------------------------
#  自動リンク
#-----------------------------------------------------------
sub auto_link {
	my $text = shift;
	
	$text =~ s/(s?https?:\/\/([\w-.!~*'();\/?:\@=+\$,%#]|&amp;)+)/<a href="$1" target="_blank">$1<\/a>/g;
	return $text;
}

#-----------------------------------------------------------
#  エラー画面
#-----------------------------------------------------------
sub error {
	my $err = shift;
	
	open(IN,"$cf{tmpldir}/error.html") or die;
	my $tmpl = join('',<IN>);
	close(IN);
	
	$tmpl =~ s/!cmnurl!/$cf{cmnurl}/g;
	$tmpl =~ s/!error!/$err/g;
	
	print "Content-type: text/html; charset=utf-8\n\n";
	print $tmpl;
	exit;
}

#-----------------------------------------------------------
#  繰越ボタン作成
#-----------------------------------------------------------
sub make_pgbtn {
	my ($i,$pg) = @_;
	
	# ページ繰越定義
	$cf{pg_max} ||= 10;
	my $next = $pg + $cf{pg_max};
	my $back = $pg - $cf{pg_max};
	
	# ページ繰越ボタン作成
	my $pg_btn;
	if ($back >= 0 || $next < $i) {
		$pg_btn .= "Page: ";
		
		my ($x,$y) = (1,0);
		while ($i > 0) {
			if ($pg == $y) {
				$pg_btn .= qq(| <b>$x</b> );
			} else {
				$pg_btn .= qq(| <a href="$cf{diary_cgi}?mode=$in{mode}&amp;y=$in{y}&amp;m=$in{m}&amp;pg=$y">$x</a> );
			}
			$x++;
			$y += $cf{pg_max};
			$i -= $cf{pg_max};
		}
		$pg_btn .= "|";
	}
	return $pg_btn;
}

#-----------------------------------------------------------
#  画像表示
#-----------------------------------------------------------
sub image {
	my ($i,$no,$ex,$w,$h) = @_;
	
	my $image;
	if (-f "$cf{imgdir}/$no-s-$i$ex") {
		$image = qq|<img src="$cf{imgurl}/$no-s-$i$ex" class="img" alt="">|;
	
	} else {
		($w,$h) = resize($w,$h);
		$image = qq|<img src="$cf{imgurl}/$no-$i$ex" width="$w" height="$h" class="img" alt="">|;
	}
	return qq|<a href="$cf{imgurl}/$no-$i$ex" target="_blank">$image</a>\n|;
}

